<?php
session_start();

// Path ke file JSON — sesuaikan jika perlu
$jsonPath = __DIR__ . '/urls.json';

// helper: baca JSON (return assoc array) atau false on error
function read_json_file($path) {
    if (!file_exists($path)) return false;
    $s = file_get_contents($path);
    if ($s === false) return false;
    $data = json_decode($s, true);
    return json_last_error() === JSON_ERROR_NONE ? $data : false;
}

// helper: safe write (backup + atomic write)
function write_json_file_atomic($path, $data) {
    $dir = dirname($path);
    $tmp = tempnam($dir, 'tmp_json_');
    $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    if ($json === false) return "JSON encode error: " . json_last_error_msg();

    // create backup
    $bak = $path . '.bak.' . date('YmdHis');
    if (file_exists($path)) {
        if (!copy($path, $bak)) {
            return "Failed to create backup ($bak).";
        }
    }

    if (file_put_contents($tmp, $json) === false) {
        return "Failed to write temporary file.";
    }
    // optional: set file perm
    @chmod($tmp, 0644);
    if (!rename($tmp, $path)) {
        // try to cleanup tmp
        @unlink($tmp);
        return "Failed to move temporary file into place.";
    }
    return true;
}

// CSRF token generate
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
}

$errors = [];
$success = null;
$data = read_json_file($jsonPath);
if ($data === false) {
    $errors[] = "Gagal membaca file JSON. Pastikan file '$jsonPath' ada dan dapat dibaca oleh webserver.";
    // provide a minimal default
    $data = ['allowed_url' => '', 'denied_url' => ''];
}

// Handle POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF check
    $token = $_POST['csrf_token'] ?? '';
    if (!hash_equals($_SESSION['csrf_token'], (string)$token)) {
        $errors[] = "Token CSRF tidak valid. Coba muat ulang halaman.";
    } else {
        $new_allowed = trim($_POST['allowed_url'] ?? '');
        $new_denied = trim($_POST['denied_url'] ?? '');
        
        // Validate allowed_url
        if ($new_allowed === '') {
            $errors[] = "Allowed URL tidak boleh kosong.";
        } else if (!filter_var($new_allowed, FILTER_VALIDATE_URL)) {
            $errors[] = "Format Allowed URL tidak valid. Pastikan termasuk scheme (https://...).";
        }
        
        // Validate denied_url
        if ($new_denied === '') {
            $errors[] = "Denied URL tidak boleh kosong.";
        } else if (!filter_var($new_denied, FILTER_VALIDATE_URL)) {
            $errors[] = "Format Denied URL tidak valid. Pastikan termasuk scheme (https://...).";
        }
        
        // If no errors, update and save
        if (empty($errors)) {
            $data['allowed_url'] = $new_allowed;
            $data['denied_url'] = $new_denied;
            
            $r = write_json_file_atomic($jsonPath, $data);
            if ($r === true) {
                $success = "Allowed URL dan Denied URL berhasil diperbarui.";
                // refresh token to prevent double submit
                $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
            } else {
                $errors[] = "Gagal menyimpan file: $r";
            }
        }
    }
}

// For display
$current_allowed = htmlspecialchars($data['allowed_url'] ?? '', ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
$current_denied  = htmlspecialchars($data['denied_url'] ?? '', ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
$csrf_token = $_SESSION['csrf_token'];
?>
<!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8">
  <title>Edit URLs — urls.json</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body{font-family:system-ui,Segoe UI,Roboto,Arial;margin:24px;background:#f7f8fa;color:#222}
    .card{background:#fff;padding:18px;border-radius:8px;max-width:820px;margin:0 auto;box-shadow:0 6px 18px rgba(0,0,0,.06)}
    label{display:block;margin:12px 0 6px;font-weight:600}
    input[type="text"]{width:100%;padding:10px;border:1px solid #ddd;border-radius:6px}
    .btn{display:inline-block;margin-top:12px;padding:10px 14px;border-radius:6px;background:#0b74de;color:#fff;text-decoration:none;border:none;cursor:pointer}
    .err{background:#ffefef;border:1px solid #f1c0c0;padding:8px;border-radius:6px;margin-bottom:10px;color:#900}
    .ok{background:#eef7ed;border:1px solid #c7e6c7;padding:8px;border-radius:6px;margin-bottom:10px;color:#196400}
    .meta{font-size:13px;color:#666;margin-top:8px}
    .small{font-size:13px;color:#555}
  </style>
</head>
<body>
  <div class="card">
    <h2>Edit URLs</h2>

    <?php if ($errors): ?>
      <div class="err">
        <?php foreach ($errors as $e) echo '<div>'.htmlspecialchars($e).'</div>'; ?>
      </div>
    <?php endif; ?>

    <?php if ($success): ?>
      <div class="ok"><?=htmlspecialchars($success)?></div>
    <?php endif; ?>

    <form method="post" novalidate>
      <label for="allowed_url">Allowed URL</label>
      <input id="allowed_url" name="allowed_url" type="text" value="<?= $current_allowed ?>" placeholder="https://example.com/?verified" required>

      <label for="denied_url">Denied URL</label>
      <input id="denied_url" name="denied_url" type="text" value="<?= $current_denied ?>" placeholder="https://example.com/" required>

      <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf_token, ENT_QUOTES, 'UTF-8') ?>">
      <button class="btn" type="submit">Simpan perubahan</button>
    </form>

    <p class="meta">File yang diedit: <code><?= htmlspecialchars($jsonPath) ?></code></p>
    <p class="meta">Backup otomatis dibuat sebelum penulisan (format: <code>urls.json.bak.YYYYmmddHHMMSS</code>).</p>
  </div>
</body>
</html>